#ifdef WINDOWS

#include "winsock2.h"

#else

#include <sys/types.h>
#include <sys/socket.h>
#include <time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>

typedef int SOCKET;
#define SD_BOTH 2
#define SOCKET_ERROR -1
#define closesocket(a) close(a)
#define Sleep(a) sleep(a)

#endif

#include "stdio.h"
#include "time.h"
#include "setjmp.h"
#include "sysinc.h"
#include "kernel_if.h"
#include "dbgcmds.h"


extern jmp_buf mark; 
extern const unsigned long dbgTabSize;
extern const unsigned long dbgTab[];
extern unsigned int        CurrentNoOfProcesses;
extern msg_t               queue[];
extern msg_t*              queueReadPtr;
extern msg_t*              queueWritePtr;
extern msg_t*              lastQueueEntry;
extern timerEntry_t*       pFirstTimer;        
extern procTabEntry_t      schedTab[MaxNoOfProcesses];
extern char                mscbuff[0x80];
extern pID_t               curpid;   


extern void SetMscMsgTraceInfo(unsigned int deliverstate,unsigned long msgid,unsigned long msgData,pID_t senderPid);
extern void SetMsgTraceImplicitConsumed(int err);

int               bStopped;
int               bNoDebug;
int               bMsc;
unsigned long     traceTick;
unsigned long     remtick;
timerval_t        breaktime;

static SOCKET     dbgsock=0;
static int        bConnected;
static struct     sockaddr_in from;  
static int        addrlen;
static time_t     lastping;

static int HandleDbgMessage( char* msgtxt ,int txtlen);
static void BreakCallBack(unsigned long symbolId);
void DoRun(unsigned long symbolId);
void DoStep(unsigned long symbolId);
static void UpdateTimerValues(timerval_t* pBreaktime);
static void SetProcInfo(unsigned char* buff,int index);  
static void SetTimerInfo(unsigned char* buff,timerEntry_t* pt);
static void SetMsgInfo(unsigned char* buff,msg_t* pm);
static void SetSystemInfo(unsigned char* buff);



const unsigned char fClose[]={D_TCLOSE,0};
const unsigned char fTRun[]={D_TRUN,0};
const unsigned char fTStop[]={D_TSTOP,0};


int InitDbgSocket()
{
struct sockaddr_in sa;
int err;
struct hostent* pht;
char buff[100];
unsigned long dontblock;

  if(dbgsock==0)
  {

#ifdef WINDOWS
    WSADATA wsaData;
    unsigned short wVersionRequested;

    wVersionRequested = MAKEWORD( 2, 2 ); 
    err = WSAStartup( wVersionRequested, &wsaData );
    if ( err != 0 )
    {
      /* we could not find a usable WinSock DLL. */
      return 0;
    }
#endif
    
reconnect:
    /* Create the socket.*/
    if ((dbgsock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) 
    {
      printf("cannot create socket\n");
      return 0;
    }
  
    sa.sin_family = AF_INET;
    sa.sin_port = htons(DBGPORT);
    
    gethostname(buff,sizeof(buff));
    pht=gethostbyname(buff);  
    memcpy(&sa.sin_addr.s_addr, pht->h_addr_list[0],sizeof(sa.sin_addr.s_addr));
  
    sa.sin_addr.s_addr=0;
    memset(sa.sin_zero, 0, sizeof(sa.sin_zero));
  
    /* Bind the socket. */
    err = bind(dbgsock, (struct sockaddr*)&sa, sizeof(sa));
    if(err < 0) 
    {
      shutdown(dbgsock,SD_BOTH);
      closesocket(dbgsock);
      printf("cannot bind socket\n");
      bConnected=0;
      return 0;
    }
    dontblock=1;
#ifdef WINDOWS  
    ioctlsocket(dbgsock,FIONBIO,&dontblock);
#else
    ioctl(dbgsock,FIONBIO,&dontblock);
#endif
    bConnected=0;
    addrlen=sizeof(from);
    while(bConnected==0)
    {
      err=recvfrom(dbgsock,buff,sizeof(buff),0,(struct sockaddr*)&from,&addrlen); 
      if( err>0 )  
      {
        if(buff[0]=D_PINGD)
        {
          bConnected=1; 
          buff[0]=D_RUN;
          buff[1]=0;
          SendFrame(buff);    
          time(&lastping);    
          break;
        } 
      }
      else if(err==SOCKET_ERROR ) /*the socket has been closed*/
      {
#ifdef WINDOWS
        err=WSAGetLastError();
        if(err != WSAEWOULDBLOCK)
        {
          printf("    SOCKET_ERROR: %d=%X\n",err,err); 
          /*try to reconnect*/  
          shutdown(dbgsock,SD_BOTH);
          closesocket(dbgsock);
          goto reconnect;  
        }
#else        
        if(errno != EAGAIN) 
        {
          printf("    SOCKET_ERROR: %d=%X\n",errno,errno); 
          /*try to reconnect*/  
          shutdown(dbgsock,SD_BOTH);
          closesocket(dbgsock);
          goto reconnect;  
        }
#endif
      }
#ifdef WINDOWS
      Sleep(200);   /*200ms*/ 
#endif
    }
  }
  return dbgsock; 
}  

void InitDebugTable( void(*fct)(unsigned long) )
{
unsigned int i;
   
  dbg[0]=RunTimeError;
  for(i=1;i<dbgTabSize;i++)
    dbg[i]=fct;
  bMsc=0;  /* MSC default off */
}

int Debug()
{
char   buff[200]; 
int rc, retcode;
fd_set fdr;
struct timeval tval={0,0};
time_t curtime;
   
  if(bNoDebug==TRUE)
   return 0;
  rc=0;
  retcode=0; 
  time(&curtime);
  if(curtime-lastping >=4) //no ping in the last 4 second
  {    
    bConnected=0; 
     /*try to reconnect*/  
    shutdown(dbgsock,SD_BOTH);
    closesocket(dbgsock);
    dbgsock=0;
    InitDbgSocket();
    if(!bStopped)
      return 0;
  }
  while(retcode == 0)
  {
    if(bConnected == 1)
    {  
#ifdef WINDOWS  
      fdr.fd_array[0]=dbgsock;
      fdr.fd_count=1;
#else
      FD_SET(dbgsock,&fdr);
      tval.tv_sec = 1;
      tval.tv_usec= 0;
#endif
      rc=select(dbgsock+1 ,&fdr,0,0,&tval);
      if(rc == 1)
      {  
        rc=recvfrom(dbgsock,buff,sizeof(buff),0,(struct sockaddr*)&from,&addrlen); 
        if( rc>0 )  
        { 
          /*Handle the received message*/
          retcode=HandleDbgMessage(buff,rc);
          if(retcode==RUNSDL)
          {
            UpdateTimerValues(&breaktime); 
            return retcode;                             
          }
        }
        else if(rc==-1) /*the socket has been closed*/
        {
          bConnected=0; 
           /*try to reconnect*/  
          shutdown(dbgsock,SD_BOTH);
          closesocket(dbgsock);
          InitDbgSocket(); 
        }
      }  
      else if(!bStopped)
      {
        break;
      } 
    }  
    else
    {
      bConnected=0; 
       /*try to reconnect*/  
      shutdown(dbgsock,SD_BOTH);
      closesocket(dbgsock);
      dbgsock=0;
      InitDbgSocket(); 
    } 
#ifdef WINDOWS
      Sleep(200);   /*200ms*/ 
#endif
  } 
  return retcode;
}  

void SendString(char* txt)
{
  /*printf(">>type=%d len=%d\n",txt[0],txt[1]);*/
#ifdef _DBGSOCKET
  sendto(dbgsock,txt,strlen(txt)+1,0,(struct sockaddr*)&from,addrlen);
#else
   /*serial comes later*/ 
#endif
}

void SendFrame(unsigned char* buffer)
{
#ifdef _DBGSOCKET
  sendto(dbgsock,buffer,buffer[1]+2,0,(struct sockaddr*)&from,addrlen);
  #ifdef WINDOWS
      Sleep(10);   /*10ms*/ 
  #endif
  /*printf("%X\n",traceTick);*/ 
#else
   /*serial comes later*/ 
#endif
}

void CloseSocket()
{
  shutdown(dbgsock,SD_BOTH);
  closesocket(dbgsock);
  dbgsock=0;
}

/*
#>getmem(start,size)
>getCurrentProcess()  //state,timers,messages
#>setBreak(symid)
>getAllTimers()      //timerlist
#getAllProcesses     //processlist
#<bkpt(symid)
#>go()
#>break()
#>step()
#>delBkpt(symid)
#>delAllBkpts()
>setTimer(timIdx)
>sendMsg(msgId)
>setState(procId,state)
>setMem(start,len,charString)
#>ping()
#<pong()
#>restartSdl()
#>exitSdl()
>startProcess(processName,instanceNo)
>deleteProcess(procId)
*/ 

char testbuff[0x1000];

int HandleDbgMessage( char* msgtxt ,int txtlen)
{  
unsigned long symid;
unsigned short i,n;
unsigned char buff[256];
unsigned long addr;
timerEntry_t* pt;
msg_t*        pmsg;
msgId_t       msgID;
msgEntry_t    msgPar;
unsigned short receiverpid;
unsigned short senderpid;

  switch(msgtxt[0])
  {
    case D_PINGD:
    {
       /*TRACE("pingd\n");*/ 
      msgtxt[1]=0;
      if(bStopped==TRUE) 
        SendFrame((unsigned char*)fTStop);
      else
        SendFrame((unsigned char*)fTRun);     
      time(&lastping);
      return 0;
    }
    case D_EXITSDL:
    {
      /* TRACE("exitsdl\n");*/ 
      SendFrame((unsigned char*)fClose); 
      CloseSocket();
      longjmp(mark,TERMSDL); 
      return TERMSDL;
    }
    case D_RESTART:
    {  
       /*TRACE("restartsdl\n");*/ 
      SendFrame((unsigned char*)fTStop);
      bStopped=FALSE;
      longjmp(mark,RESTARTSDL); 
      return RESTARTSDL;
    }
    case D_BREAK:
    { 
       /*TRACE("break\n");*/ 
      SendFrame((unsigned char*)fTStop);
      bStopped=TRUE;    
      return 0;
    }
    case D_RUN:
    {
      /* TRACE("run\n");*/ 
      SendFrame((unsigned char*)fTRun);  
      bStopped=FALSE;
      ResetDbgStepMode();
      return RUNSDL;
    }
    case D_SETBREAK:
    {
      msgtxt[1]=5;
      symid=*(unsigned long*)&msgtxt[2];
       /*TRACE("setbreak\n");*/ 
      for(i=0;i<dbgTabSize;i++) 
      {
        if(dbgTab[i]==symid)
        {
          dbg[i]=BreakCallBack;
          msgtxt[6]=1;           
          SendFrame(msgtxt);  /*acknowledge ok*/ 
          return 0;
        }
      }
      msgtxt[6]=0; 
      SendFrame(msgtxt);  /*acknowledge failed*/ 
    }
    case D_DELALLBRK:
    {
       /*TRACE("delallbreaks\n");*/ 
      InitDebugTable(DoStep);
      SendFrame(msgtxt);  /*acknowledge ok*/ 
      return 0;
    }
    case D_DELBREAK:
    {      
      symid=*(unsigned long*)&msgtxt[2];
       /*TRACE("delbreak\n");*/ 
      for(i=0;i<dbgTabSize;i++) 
      {
        if(dbgTab[i]==symid)
        {
          dbg[i]=DoRun;
          break;
        }
      } 
      SendFrame(msgtxt);  /*acknowledge ok*/ 
      return 0;
    }
    case D_STEP:
    {
       /*TRACE("step\n");*/ 
      for(i=0;i<dbgTabSize;i++) 
      {
        if(dbg[i]==DoRun)
        {
          dbg[i]=DoStep;
        }
      } 
      bStopped=FALSE;
      return RUNSDL;
    }
    case D_INIT:
    {
       /*TRACE("init\n");*/  
      SendFrame(msgtxt);  /*acknowledge ok*/ 
      return 0;
    }
    case D_GETMEM:
    {            
      if(msgtxt[1]==12)  /*important at this moment*/
      {
        addr=(unsigned char)msgtxt[2]<<24;
        addr|=(unsigned char)msgtxt[3]<<16;
        addr|=(unsigned char)msgtxt[4]<<8;
        addr|=(unsigned char)msgtxt[5];
        /*msgtxt[11]=memspec currently not used*/
        i=(unsigned short)(msgtxt[10]<<8);
        i|=(unsigned char)msgtxt[11];
        buff[0]=D_GETMEM;
        while(i)
        {
          if(i>=240)
            n=240;
          else 
            n=i;
          buff[1]=n+9;
          buff[2]=(unsigned char)(addr >> 24) &0xFF;
          buff[3]=(unsigned char)(addr >> 16) &0xFF;
          buff[4]=(unsigned char)(addr >> 8) &0xFF;
          buff[5]=(unsigned char)addr &0xFF;
          memcpy(&buff[6],&msgtxt[6],4); /*the window ID*/
          buff[10]=msgtxt[6] & 0x7F;     /*currently not used the memspec*/
          
#ifdef WINDOWS          
         __try
         {  
           unsigned char* pc1;
           unsigned char* pc2;
           int nbytes;
           
           nbytes=n;  
           pc1=&buff[11];
           pc2=(unsigned char*)addr; 
           n=0;
           while( n < nbytes)
           {
             *pc1=*pc2;
             pc1++;
             pc2++;
             n++;  
           }           
         }
         __except( EXCEPTION_EXECUTE_HANDLER )
         {
           buff[10]|=0x80;                /*set it invalid*/
           buff[1]=9+n;
           i=0;
           /*TRACE("read from invalid memory\n");*/
         } 
#else
         memcpy(&buff[11],(unsigned char*)addr,n);        
#endif               
         i-=n;
         SendFrame(buff);  /*acknowledge ok*/
         addr+=n;          
          /*TRACE("mem\n");*/
        }         
        break;
      }
      else
      {
        SendFrame(msgtxt); /*not served request*/
      }
      break;
    }
    case D_SETMEM:
    {
      if(msgtxt[1]==8)  /*important at this moment*/
      {
        addr=(unsigned char)msgtxt[2]<<24;
        addr|=(unsigned char)msgtxt[3]<<16;
        addr|=(unsigned char)msgtxt[4]<<8;
        addr|=(unsigned char)msgtxt[5];
        /*msgtxt[6]=memspec currently not used*/
        *(unsigned char*)addr=msgtxt[7];
        //TRACE("setmem\n");
      } 
      break;
    }
    case D_GETSYS:
    {
      SetSystemInfo(buff);      
      SendFrame(buff);   //send the summary
      break; 
    }
    case D_GETPROCESS:
    {
      if(msgtxt[1]==0)
      {
        if( CurrentNoOfProcesses )
        { 
          for(i=0;i<CurrentNoOfProcesses;i++)
          {
            if(schedTab[i].proc)
            { 
              SetProcInfo(buff,i);
              SendFrame(buff);            
            }
          }
        }
        else
        {
          SetProcInfo(buff,-1);
          SendFrame(buff);   //send the information for this process                        
        }       
        break; 
      }
      else
      {
        memcpy(buff,&msgtxt[2],msgtxt[1]);
        buff[msgtxt[2]]=0;
        for(i=0;i<CurrentNoOfProcesses;i++)
        {
          if(!strcmp(schedTab[i].thisProcess.processname,buff))
          { 
            SetProcInfo(buff,i);           
            SendFrame(buff);   //send the information for this process
            break; 
          }
        }
      }
    }
    case D_GETTIMER:
    {
      i=0;
      if(pFirstTimer)
      {
        pt=pFirstTimer; 
        while(pt!=0)
        {
          i++;
          SetTimerInfo(buff,pt);        
          pt=pt->pNext;
          SendFrame(buff);   //send the information for this timer 
        } 
      }
      else
      {
        SetTimerInfo(buff,NULL); 
        SendFrame(buff); 
      }
      break;
    }
    case D_GETMSG:
    {
      pmsg=queueReadPtr;
      if(queueReadPtr!=queueWritePtr)
      {
        while(pmsg!=queueWritePtr)
        {
          SetMsgInfo(buff,pmsg); 
          SendFrame(buff);   //send the information for this message entry 
          pmsg++;  
          if(pmsg>=lastQueueEntry)
            pmsg=queue;
        }
      }
      else
      {
        SetMsgInfo(buff,NULL); 
        SendFrame(buff); 
      }
      break;
    }
    case D_STARTMSC:
    {
      SetSystemInfo(buff);      
      SendFrame(buff);
      for(i=0;i<CurrentNoOfProcesses;i++)
      {
        if(schedTab[i].proc)
        { 
          SetProcInfo(buff,i);
          SendFrame(buff);            
        }
      }
      buff[0]=D_MSC_CFGFINISHED;
      buff[1]=0; 
      SendFrame(buff); 
      bMsc=MSCENABLE;
      break;
    }
    case D_STOPMSC:
    {
      SendFrame(msgtxt);  /*acknowledge ok*/ 
      bMsc=0;
      break;
    }
    case D_INJECTMSG:
    {
      msgID=(unsigned char)msgtxt[2]<<24;
      msgID|=(unsigned char)msgtxt[3]<<16;
      msgID|=(unsigned char)msgtxt[4]<<8;
      msgID|=(unsigned char)msgtxt[5];
      receiverpid=(unsigned char)msgtxt[6]<<8;
      receiverpid|=(unsigned char)msgtxt[7];
      senderpid=(unsigned char)msgtxt[8]<<8;
      senderpid|=(unsigned char)msgtxt[9];
      msgPar=(unsigned char)msgtxt[10]<<24;
      msgPar|=(unsigned char)msgtxt[11]<<16;
      msgPar|=(unsigned char)msgtxt[12]<<8;
      msgPar|=(unsigned char)msgtxt[13];
      if(receiverpid==0xFFFF)
      {
        queueWritePtr->msgID     = msgID;
        queueWritePtr->receiverPid = (unsigned short)(msgID >>16); ;
      }
      else
      {
        queueWritePtr->msgID     = msgID&0xFFFF;
        queueWritePtr->msgID    |= (receiverpid<<16);
        queueWritePtr->receiverPid = receiverpid;
      }
      queueWritePtr->msg         = msgPar;
      if(senderpid!=0xFFFF)
      {
        queueWritePtr->senderPid = senderpid;
      }
      else
      {
        queueWritePtr->senderPid = curpid;
      }
      queueWritePtr++;
      if(queueWritePtr>=lastQueueEntry)
      {
        queueWritePtr=queue;
      }  
      if(queueWritePtr==queueReadPtr) /* in this case the queue has an overflow */
      {
        RunTimeError(ERR_QUEUEOVERFLOW); 
        break;
      }
      if(bMsc!=0)
      {
        SetMscMsgTraceInfo(D_MSC_SENDMSG,msgID,(unsigned long)msgPar,senderpid);
        traceTick++;
        SendFrame(mscbuff);
      }
      break;

    }
    default: /*unknown protocol*/
      break;
  }
  return 0;
}


void SetSystemInfo(unsigned char* buff)
{
int i;
timerEntry_t* pt;

  buff[0]=D_GETSYS;
  buff[1]=12;
  buff[2]=(CurrentNoOfProcesses-1) >> 8;      /*number of currently running processes (h)*/
  buff[3]=(CurrentNoOfProcesses-1) & 0xFF;     /*number of currently running processes (l)*/
  buff[4]=MaxNoOfProcesses >> 8;           /*max number of processes (h)*/
  buff[5]=MaxNoOfProcesses & 0xFF;         /*max number of processes (l)*/
  i=0;
  pt=pFirstTimer;
  while(pt!=0)
  {
    i++;
    pt=pt->pNext;
  } 
  buff[6]= i >> 8;                        /*number of currently running timers (h)*/ 
  buff[7]= i & 0xFF;                      /*number of currently running timers (l)*/
  buff[8]=MaxNoOfTimerEntries >> 8;       /*max number of timers (h)*/
  buff[9]=MaxNoOfTimerEntries & 0xFF;     /*max number of timers (l)*/
  if(queueWritePtr > queueReadPtr)
  {
    i=queueWritePtr - queueReadPtr;
  }
  else
  {
    i=(queueReadPtr-queue) + MaxNoOfQueueEntries - (queueWritePtr-queue);
  }
  buff[10]= i >> 8;                      /*number of currently waiting messages (h)*/ 
  buff[11]= i & 0xFF;                    /*number of currently waiting messages (l)*/
  buff[12]= MaxNoOfQueueEntries >> 8;    /*max number of messages=queue depth (h)*/
  buff[13]= MaxNoOfQueueEntries & 0xFF;  /*max number of messages=queue depth (l)*/
}


void SetMsgInfo(unsigned char* buff,msg_t* pm)
{
unsigned long tmp;
    
  buff[0]=D_GETMSG;
  if(!pm)
  {
    buff[1]=0;
  }
  else
  {  
    buff[1]=16;
    tmp=(unsigned long)pm->msgID;
    buff[2]=(unsigned char) (tmp >> 24) & 0xFF; /*the Message id*/
    buff[3]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[4]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[5]=(unsigned char) tmp & 0xFF;
    tmp=(unsigned long)pm->msg;
    buff[6]=(unsigned char) (tmp >> 24) & 0xFF; /*the message data*/
    buff[7]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[8]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[9]=(unsigned char) tmp & 0xFF;
    tmp=(unsigned long)pm->senderPid;
    buff[10]=(unsigned char) (tmp >> 24) & 0xFF; /*the sender process id*/
    buff[11]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[12]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[13]=(unsigned char) tmp & 0xFF;
    tmp=(unsigned long)pm->receiverPid;
    buff[14]=(unsigned char) (tmp >> 24) & 0xFF; /*the receiver process id*/
    buff[15]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[16]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[17]=(unsigned char) tmp & 0xFF;
  }
}

void SetTimerInfo(unsigned char* buff,timerEntry_t* pt)
{
unsigned long tmp;
timerval_t dt;

  buff[0]=D_GETTIMER;
  if(!pt)
  {
    buff[1]=0;
  }
  else
  {
    buff[1]=14;
    buff[2]=(unsigned char) (pt->msgId >> 8) & 0xFF; 
    buff[3]=(unsigned char) (pt->msgId & 0xFF);

    tmp=(unsigned long)pt->sendByPid;
    buff[4]=(unsigned char) (tmp >> 8) & 0xFF;/*the process id*/
    buff[5]=(unsigned char) tmp & 0xFF;  

    GetTimeDiff(&pt->t,&breaktime,&dt);
    tmp=(unsigned long)dt;    
    buff[6]=sizeof(timerval_t);
    buff[7]=(unsigned char) (tmp >> 24) & 0xFF;
    buff[8]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[9]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[10]=(unsigned char) tmp & 0xFF;
    tmp=(unsigned long)pt->t;
    buff[11]=sizeof(timerval_t);
    buff[12]=(unsigned char) (tmp >> 24) & 0xFF;
    buff[13]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[14]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[15]=(unsigned char) tmp & 0xFF;
    
  }
}


void SetProcInfo(unsigned char* buff,int index)
{
unsigned long tmp;

  buff[0]=D_GETPROCESS;           
  if(index!=-1)
  {
    tmp=(unsigned long)schedTab[index].thisProcess.procid;         
    buff[2]=(unsigned char) (tmp >> 24) & 0xFF; /*the process id*/
    buff[3]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[4]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[5]=(unsigned char) tmp & 0xFF;
    tmp=(unsigned long)schedTab[index].thisProcess.instanceCnt;
    buff[6]=(unsigned char) (tmp >> 24) & 0xFF;
    buff[7]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[8]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[9]=(unsigned char) tmp & 0xFF; 
    tmp=(unsigned long)schedTab[index].thisProcess.state;
    buff[10]=(unsigned char) (tmp >> 24) & 0xFF;
    buff[11]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[12]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[13]=(unsigned char) tmp & 0xFF;
    tmp=(unsigned long)schedTab[index].thisProcess.pPrivatData; 
    buff[14]=(unsigned char) (tmp >> 24) & 0xFF;
    buff[15]=(unsigned char) (tmp >> 16) & 0xFF;
    buff[16]=(unsigned char) (tmp >> 8) & 0xFF;
    buff[17]=(unsigned char) tmp & 0xFF;
    tmp=strlen(schedTab[index].thisProcess.processname)+1;
    if(tmp<230)
      buff[18]=(unsigned char) tmp;
    else
      buff[18]=230;
    memcpy(&buff[19],schedTab[index].thisProcess.processname,tmp);
  }
  else
  {
    memset(&buff[2],0,16);
    buff[18]=sizeof("Not initialized");
    strcpy(&buff[19],"Not initialized");
  }
  buff[1]=17+buff[18];
}

void DoRun(unsigned long symboId)
{
  if((bMsc & (MSGNOTIFY+MSCENABLE))==MSCENABLE) 
  {
    SetMsgTraceImplicitConsumed(NOERROR);           
    SendFrame(mscbuff);
    bMsc|=MSGNOTIFY;
  }
  return;
}

void DoStep(unsigned long symbolId)
{
char buff[10];
int rc;

   if((bMsc & (MSGNOTIFY+MSCENABLE))==MSCENABLE) 
   {
     SetMsgTraceImplicitConsumed(NOERROR);           
     SendFrame(mscbuff);
     bMsc|=MSGNOTIFY;
   }
   GetCurTime(&breaktime);
   buff[0]=D_BREAKREACHED;
   buff[1]=4;
   buff[2]=(unsigned char)(symbolId&0xFF);
   buff[3]=(unsigned char)((symbolId>>8)&0xFF);
   buff[4]=(unsigned char)((symbolId>>16)&0xFF);
   buff[5]=(unsigned char)(symbolId>>24);
   SendFrame(buff);
   bStopped=TRUE;
   TRACE("Next reached\n");
   rc=Debug();    
   if(rc == TERMSDL || rc==RESTARTSDL)
   {
     longjmp(mark,rc);
   }
   return;
}

void BreakCallBack(unsigned long symbolId)
{
char buff[10];
int rc;

   GetCurTime(&breaktime);
   buff[0]=D_BREAKREACHED;
   buff[1]=4;
   *(unsigned long*)&buff[2]=symbolId;
   SendFrame(buff);
   bStopped=TRUE;
   TRACE("Breakpoint reached\n");
   rc=Debug();    
   if(rc == TERMSDL || rc==RESTARTSDL)
   {
     longjmp(mark,rc);
   }
   return;
}

void UpdateTimerValues(timerval_t* pBreaktime)
{
timerval_t starttime,difftime;
timerEntry_t* pt;

  GetCurTime(&starttime);
  GetTimeDiff(&starttime,pBreaktime,&difftime);
  pt=pFirstTimer;
  while( pt != 0)
  {  
    AddTimeVals(&pt->t,&difftime);
    pt=pt->pNext;
  }
}

void TRACE(char* TraceText)
{
char buffer[256];
int len;

  buffer[0]=D_TRACE;  
  len=strlen(TraceText);
  if(len>253)
  {
    len=253;
    TraceText[253]=0;  /*shorten it*/
  }
  buffer[1]=strlen(TraceText)+1;
  strcpy(&buffer[2],TraceText);
  SendFrame(buffer);
}

void ResetDbgStepMode()
{
unsigned int i;

  for(i=0;i<dbgTabSize;i++) 
  {
    if(dbg[i]==DoStep)
    {
      dbg[i]=DoRun;
    }
  } 
}

