/*
 * JSDL kernel:
 * Implementation of the JSDL run-time system
 * Originally written by Jens Altmann
 *
 * Enhancements by donGerry:
 *
 * code for 8-bit ATmega CPUs (AVR), no 32-bit integers!
 *
 * $Id: jsdlkern.c 53 2008-10-21 19:03:44Z gerry $
 */

#if (defined(__AVR) && defined(_SDLDEBUG))
#error "SDL kernel for a AVR target cannot be compiled with debug option turned on!"
#endif

#ifdef __AVR
#include "avr/io.h"
#include "app_timer.h"
#else
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <sys/timeb.h>
#include "setjmp.h"
#include <memory.h>
#endif //__AVR

#include "sysinc.h"  // this also includes 'kernel_if.h' !

#ifdef WINDOWS
#include "Windows.h"
#endif

#ifdef _SDLDEBUG
#include "dbgcmds.h"

jmp_buf mark;
extern unsigned long traceTick;
extern unsigned long remtick;
extern int bMsc;
extern int bNoDebug;
extern timerval_t  breaktime;
extern const unsigned char fClose[];
char                  mscbuff[0x80];

void SetMsgTraceImplicitConsumed(int err);
void SetMscMsgTraceInfo(unsigned int deliverstate,unsigned long msgid,unsigned long msgData,pID_t senderPid);
static void SetMscStateTraceInfo();
static void SetMscTimTraceInfo(unsigned long timId,unsigned long timeout);
static void SetMsgProcessExit();

extern void DoStep(unsigned long symbolId);
extern void DoRun(unsigned long symboId);

#endif /*_SDLDEBUG*/

#define NULLMESSAGE ((msgId_t)0)

extern processDef_t processTable[];

#ifdef __AVR
extern char bStopped;
static char bInit;
char        CurrentNoOfProcesses;
#else
extern int bStopped;
static int bInit;
int        CurrentNoOfProcesses;
#endif //__AVR

static timerEntry_t   timerTab[MaxNoOfTimerEntries];
procTabEntry_t        schedTab[MaxNoOfProcesses];
msg_t                 queue[MaxNoOfQueueEntries];
msg_t*                queueReadPtr;
msg_t*                queueWritePtr;
msg_t*                lastQueueEntry;
static timerEntry_t*  pNextFreeTimer;
timerEntry_t*         pFirstTimer;

/* a global variable that is set always at task context switch*/
pID_t                 curpid;

void  InitTimers()
{
#ifdef __AVR
register unsigned char i;
#else
unsigned int i;
#endif //__AVR

  for(i=0;i<MaxNoOfTimerEntries;i++)
  {
    if(i<MaxNoOfTimerEntries-1)
    {
      timerTab[i].pNext=&timerTab[i+1];
    }
    else
    {
      timerTab[i].pNext=(timerEntry_t*)0;
    }
    timerTab[i].t=(timerval_t)-1;
    timerTab[i].msgId=NULLMESSAGE;
    timerTab[i].sendByPid=0;
  }
  pFirstTimer=(timerEntry_t*)0;
  pNextFreeTimer=&timerTab[0];
}

#ifdef __AVR
char jsdlInit()
{
register unsigned char i;
register char          rc;
#else
int jsdlInit()
{
unsigned int i;
int          rc;
#endif //__AVR
msg_t dummyMsg;

  bInit=1;
  //initialize the process table
  for(i=0;i<MaxNoOfProcesses;i++)
  {
    schedTab[i].proc=0;
    schedTab[i].thisProcess.procid=0;
    schedTab[i].thisProcess.instanceCnt=0;
    schedTab[i].thisProcess.state=INITSTATE;
    schedTab[i].thisProcess.pPrivatData=0;
  }
  //set the init - process 0 is reserved for system usage
  for(i=1;i<=NoOfProcesses;i++)
  {
    schedTab[i].proc=processTable[i-1].processEntry;
#ifndef __AVR
    schedTab[i].thisProcess.processname=processTable[i-1].processName;
#endif
    schedTab[i].thisProcess.procid=i;
  }
  CurrentNoOfProcesses=NoOfProcesses+1;
  //Initialize the queue
#ifndef __AVR
  memset(queue,0,sizeof(queue));
#endif
  queueReadPtr=queue;
  queueWritePtr=queue;
  lastQueueEntry=&queue[MaxNoOfQueueEntries];
  //Initialize the Timer
  InitTimers();

  dummyMsg.msgID=NULLMESSAGE;
  //start the processes
  for(curpid=1;curpid<CurrentNoOfProcesses;curpid++)
  {
    schedTab[curpid].thisProcess.pMsg=&dummyMsg;
    rc=(*schedTab[curpid].proc)(&schedTab[curpid].thisProcess);
    if(rc==-1)  /*the process has been finished*/
    {
      schedTab[curpid].proc=0;
      CurrentNoOfProcesses--;
      if(CurrentNoOfProcesses==1)
      {
        return TERMSDL;
      }
    }
  }
  return 0;
}

#ifdef __AVR
char JsdlCycle()
{
register char rc;
#else
int JsdlCycle()
{
int rc;
unsigned int prevstate;
#endif //__AVR
timerEntry_t* pt;
timerval_t curTime;
#ifdef _SDLDEBUG
register msgId_t    msgid;
register msgEntry_t msgparam;

  GetCurTime(&breaktime);
  rc=Debug();
  if(rc & TERMSDL)
    return rc;
#endif

  rc = 0;
  /*first process the next pending queue entry*/
  if(queueWritePtr!=queueReadPtr)
  {
#ifdef _SDLDEBUG
    msgid=queueReadPtr->msgID;
    msgparam=queueReadPtr->msg;
#endif
    curpid=queueReadPtr->receiverPid;
    schedTab[curpid].thisProcess.pMsg=queueReadPtr;
    queueReadPtr++;
    if(queueReadPtr>=lastQueueEntry)
    {
      queueReadPtr=queue;
    }
    if(schedTab[curpid].proc)  /*call it only if it exist*/
    {
#ifdef _SDLDEBUG
      if(bMsc & MSCENABLE)
      {
        prevstate=schedTab[curpid].thisProcess.state;
        SetMscMsgTraceInfo(D_MSC_DELIVERMSG,msgid,msgparam,curpid);
        SendFrame(mscbuff);
        remtick=traceTick;
        traceTick++;
        bMsc = MSCENABLE; /*reset the notify bit*/
      }
#endif

      rc=(*schedTab[curpid].proc)(&schedTab[curpid].thisProcess);

#ifdef _SDLDEBUG
      if(bMsc & MSCENABLE)
      {
        if(prevstate!=schedTab[curpid].thisProcess.state)  /*the state has changed*/
        {
          SetMscStateTraceInfo(mscbuff);
          traceTick+=2;
          SendFrame(mscbuff);
        }
      }
#endif /*_SDLDEBUG*/

    }
    if(rc==-1)  /*the process has been finished*/
    {
#ifdef _SDLDEBUG
      if(bMsc & MSCENABLE)
      {
        SetMsgProcessExit();
        traceTick++;
        SendFrame(mscbuff);
      }
#endif /*_SDLDEBUG*/
      rc=0;
      schedTab[curpid].proc=0;
      CurrentNoOfProcesses--;
      if(CurrentNoOfProcesses==1)
      {
#ifdef _SDLDEBUG
        SendFrame((unsigned char*)fClose);
  #ifdef _DBGSOCKET
        CloseSocket();
  #endif  /*_DBGSOCKET*/
#endif  /*_SDLDEBUG*/
        return TERMSDL;
      }
    }
  }
  /* test if a timer is running down*/
  if(pFirstTimer!=(timerEntry_t*)0 )
  {
    GetCurTime(&curTime);
    while(pFirstTimer!=0)
    {
      pt=pFirstTimer;
      if(pt->t > curTime)
      {
        break; /* if the first timer is still running then all other too*/
      }
      else
      {
        queueWritePtr->msgID=pt->msgId;
        queueWritePtr->senderPid=pt->sendByPid;
        queueWritePtr->receiverPid=pt->sendByPid;
        queueWritePtr->msg=0;
        queueWritePtr++;
        if(queueWritePtr>=lastQueueEntry)
        {
          queueWritePtr=queue;
        }
        if(queueWritePtr==queueReadPtr) /* in this case the queue has an overflow */
        {
          RunTimeError(ERR_QUEUEOVERFLOW);
        }

        /*move this timer back to the free timer list*/
        pFirstTimer=pt->pNext;
        pt->pNext=pNextFreeTimer;
        pNextFreeTimer=pt;
      }
    }
  }

  if(queueWritePtr!=queueReadPtr)
  {
    rc|=MSG_AVAILABLE;
  }
  return rc;
}

#ifdef __AVR
char SetSdlTimer(char timerid, unsigned short elapseticks)
#else
int SetSdlTimer( int timerid, unsigned long elapseticks)
#endif //__AVR
{
timerval_t curTime;
register timerEntry_t* pt;
register timerEntry_t* ptInsert;

  if(pNextFreeTimer==0)
  {
    RunTimeError(ERR_NOTIMERAVAIL);
    return ERR_NOTIMERAVAIL;
  }
  pt=pFirstTimer;
  ptInsert=pt;

  GetCurTime(&curTime);
  /* seems quite obsolete to me !?
  if(pFirstTimer)
    pt=pt;
   */
  /*set values*/
  pNextFreeTimer->msgId=timerid;
  pNextFreeTimer->sendByPid=curpid;
  pNextFreeTimer->t=curTime+elapseticks;

  while(pt!=0 &&  (pt->t < pNextFreeTimer->t))
  {
    ptInsert=pt;
    pt=pt->pNext;
  }


#ifdef _SDLDEBUG
  if(bMsc & MSCENABLE)
  {
    SetMscTimTraceInfo(timerid,elapseticks);
    traceTick++;
    SendFrame(mscbuff);
  }
#endif /*_SDLDEBUG*/

  /*insert in running timer list*/
  if(ptInsert)
  {
    pt=pNextFreeTimer;
    /*remove from free timer list*/
    pNextFreeTimer=pNextFreeTimer->pNext;
    pt->pNext=ptInsert->pNext;
    ptInsert->pNext=pt;
  }
  else
  {
    pFirstTimer=pNextFreeTimer;
    /*remove from free timer list*/
    pNextFreeTimer=pNextFreeTimer->pNext;
    pFirstTimer->pNext=0;
  }
  return 0;
}

#ifdef __AVR
char KillSdlTimer(char timerid)
#else
int KillSdlTimer(int timerid)
#endif //__AVR
{
register timerEntry_t* pt;
register timerEntry_t* ptprev;

  ptprev=0;
  pt=pFirstTimer;
  while( pt != 0)
  {
    if(pt->msgId == (msgId_t)timerid)
    {
#ifdef _SDLDEBUG
      if(bMsc & MSCENABLE)
      {
        SetMscTimTraceInfo(timerid,pt->t);
        mscbuff[0]=D_MSC_KILLTIMER;
        traceTick++;
        SendFrame(mscbuff);
      }
#endif /*_SDLDEBUG*/

      /*remove from running timer list*/
      if(ptprev!=0)
      {
        ptprev->pNext=pt->pNext;
      }
      if(pt==pFirstTimer)
      {
        pFirstTimer=pt->pNext;
      }
      /*put it back in free timer list*/
      pt->pNext=pNextFreeTimer;
      pNextFreeTimer=pt;
      return NOERROR;
    }
    ptprev=pt;
    pt=pt->pNext;
  }
  return ERR_UNKNOWNTIMER;  //timer not found
}

#ifdef __AVR
char ReQueueMessage(msg_t* pMsg)
#else
int ReQueueMessage(msg_t* pMsg)
#endif //__AVR
{
  *queueWritePtr=*pMsg;
  queueWritePtr++;
  if(queueWritePtr>=lastQueueEntry)
  {
    queueWritePtr=queue;
  }
  if(queueWritePtr==queueReadPtr) /* in this case the queue has an overflow */
  {
    RunTimeError(ERR_QUEUEOVERFLOW);
    return ERR_QUEUEOVERFLOW;
  }
  return NOERROR;
}


#ifdef __AVR
char SendMsg(msgId_t msgID, msgEntry_t msgPar)
#else
int SendMsg(msgId_t msgID, msgEntry_t msgPar)
#endif //__AVR
{
  queueWritePtr->msgID       = msgID;
  queueWritePtr->msg         = msgPar;
#ifdef __AVR
  queueWritePtr->receiverPid = (pID_t)(msgID >> 8);
#else
  queueWritePtr->receiverPid = (pID_t)(msgID >> 16);
#endif //__AVR
  queueWritePtr->senderPid   = curpid;
  queueWritePtr++;
  if(queueWritePtr>=lastQueueEntry)
  {
    queueWritePtr=queue;
  }
  if(queueWritePtr==queueReadPtr) /* in this case the queue has an overflow */
  {
    RunTimeError(ERR_QUEUEOVERFLOW);
    return ERR_QUEUEOVERFLOW;
  }
#ifdef _SDLDEBUG

  if(bMsc & MSCENABLE)
  {
    SetMscMsgTraceInfo(D_MSC_SENDMSG,msgID,(unsigned long)msgPar,curpid);
    traceTick++;
    SendFrame(mscbuff);
  }

#endif /*_SDLDEBUG*/
  return 0;
}
#ifdef __AVR
char SDL_main(void)
{
register char rc;
#else
int main( int argc, char *argv[])
{
int rc;
#endif //__AVR

#ifdef _SDLDEBUG
  if(argc==1 && !strcmp(argv[0],"-r")  || argc==2 && !strcmp(argv[1],"-r"))
  {
    InitDebugTable(DoRun);
    bNoDebug=TRUE;
  }
  else
  {
    printf("This executable was build for debug and cannot\nrun without connection to JSDL!!\n");
    bStopped=TRUE;   /*stop on system start*/
    bNoDebug=FALSE;
    InitDebugTable(DoStep);
#ifdef _DBGSOCKET
    InitDbgSocket();
#else
    /*InitDebugSerial();*/
#endif
  }
#endif  /*_SDLDEBUG*/

restartSDL:;

#ifdef _SDLDEBUG
  rc=setjmp(mark);
  if(rc!=0)
    goto checkRetCode;
#endif

  rc=jsdlInit();
  if(rc & TERMSDL)
    return 0;
  while(1)
  {
    rc=JsdlCycle();
#ifdef WINDOWS
      Sleep(10);   /*10ms*/
#endif


#ifndef __AVR
checkRetCode:;
#endif
    if(rc & TERMSDL)
      return 0;
    if(rc & RESTARTSDL)
      goto restartSDL;
  }
  return 0;
}

#ifndef __AVR
void RunTimeError( unsigned long  errorcode )
{
#ifdef _SDLDEBUG
  if((bMsc & (MSGNOTIFY+MSCENABLE))==MSCENABLE)
  {
    SetMsgTraceImplicitConsumed(-1);
    SendFrame(mscbuff);
    bMsc=MSCENABLE;
  }
#endif /*_SDLDEBUG*/
  printf("Unexpected signal implicitly consumed !!\n");
}
#endif //__AVR


/*this are platform depend functions
  and must be implemented depend of it*/

void GetCurTime( timerval_t* pCurTime)
{
#ifdef __AVR
  *pCurTime = (timerval_t)get_ticks();
#else
struct timeb t;

  ftime(&t);
  *pCurTime=t.millitm+1000*t.time;
#endif //__AVR
}


/*calculates the time difference between two time
values t2-t1*/
void GetTimeDiff(timerval_t* pt2,timerval_t* pt1,timerval_t* ptdiff)
{
  *ptdiff=*pt2 - *pt1;
}

/*Adds the value t2 to t1*/
void AddTimeVals(timerval_t* pt2,timerval_t* pt1)
{
  *pt2 += *pt1;
}

#ifdef _SDLDEBUG
void SetMscMsgTraceInfo(unsigned int deliverstate,unsigned long msgid,unsigned long msgData,pID_t senderPid)
{
  memset(mscbuff,0xFF,20);

  mscbuff[0]=deliverstate;
  mscbuff[1]=16;
  mscbuff[2]=(senderPid>>24)&0xFF;
  mscbuff[3]=(senderPid>>16)&0xFF;
  mscbuff[4]=(senderPid>>8)&0xFF;
  mscbuff[5]=senderPid & 0xFF;

  mscbuff[6]=(unsigned char)(msgid>>24)&0xFF;
  mscbuff[7]=(unsigned char)(msgid>>16)&0xFF;
  mscbuff[8]=(unsigned char)(msgid>>8 )&0xFF;
  mscbuff[9]=(unsigned char)msgid & 0xFF;

  mscbuff[10]=(unsigned char)(msgData>>24)&0xFF;
  mscbuff[11]=(unsigned char)(msgData>>16)&0xFF;
  mscbuff[12]=(unsigned char)(msgData>>8 )&0xFF;
  mscbuff[13]=(unsigned char)(msgData    )&0xFF;

  mscbuff[14]=(unsigned char)(traceTick>>24)&0xFF;
  mscbuff[15]=(unsigned char)(traceTick>>16)&0xFF;
  mscbuff[16]=(unsigned char)(traceTick>>8 )&0xFF;
  mscbuff[17]=(unsigned char)traceTick & 0xFF;
}

void SetMscTimTraceInfo(unsigned long timId,unsigned long timeout)
{
  mscbuff[0]=D_MSC_SETTIMER;
  mscbuff[1]=16;
  mscbuff[2]=(curpid>>24)&0xFF;
  mscbuff[3]=(curpid>>16)&0xFF;
  mscbuff[4]=(curpid>>8)&0xFF;
  mscbuff[5]=curpid & 0xFF;

  mscbuff[6]=(unsigned char)(timId>>24)&0xFF;
  mscbuff[7]=(unsigned char)(timId>>16)&0xFF;
  mscbuff[8]=(unsigned char)(timId>>8 )&0xFF;
  mscbuff[9]=(unsigned char)timId & 0xFF;

  mscbuff[10]=(unsigned char)(timeout>>24)&0xFF;
  mscbuff[11]=(unsigned char)(timeout>>16)&0xFF;
  mscbuff[12]=(unsigned char)(timeout>>8 )&0xFF;
  mscbuff[13]=(unsigned char)(timeout    )&0xFF;

  mscbuff[14]=(unsigned char)(traceTick>>24)&0xFF;
  mscbuff[15]=(unsigned char)(traceTick>>16)&0xFF;
  mscbuff[16]=(unsigned char)(traceTick>>8 )&0xFF;
  mscbuff[17]=(unsigned char)traceTick & 0xFF;
}

void SetMscStateTraceInfo()
{
  mscbuff[0]=D_MSC_CHANGESTATE;
  mscbuff[1]=12;

  mscbuff[2]=(unsigned char)(curpid>>24)&0xFF;
  mscbuff[3]=(unsigned char)(curpid>>16)&0xFF;
  mscbuff[4]=(unsigned char)(curpid>>8 )&0xFF;
  mscbuff[5]=(unsigned char)curpid & 0xFF;

  mscbuff[6]=(unsigned char)(schedTab[curpid].thisProcess.state>>24)&0xFF;
  mscbuff[7]=(unsigned char)(schedTab[curpid].thisProcess.state>>16)&0xFF;
  mscbuff[8]=(unsigned char)(schedTab[curpid].thisProcess.state>>8 )&0xFF;
  mscbuff[9]=(unsigned char)(schedTab[curpid].thisProcess.state    )&0xFF;

  mscbuff[10]=(unsigned char)(traceTick>>24)&0xFF;
  mscbuff[11]=(unsigned char)(traceTick>>16)&0xFF;
  mscbuff[12]=(unsigned char)(traceTick>>8 )&0xFF;
  mscbuff[13]=(unsigned char)traceTick & 0xFF;
}

void SetMsgTraceImplicitConsumed(int err)
{
  if(err==NOERROR)
    mscbuff[0]=D_MSC_DELIVERMSG_OK;
  else
    mscbuff[0]=D_MSC_DELIVERMSG_ERR;
  mscbuff[1]=4;
  mscbuff[2]=(unsigned char)(remtick>>24)&0xFF;
  mscbuff[3]=(unsigned char)(remtick>>16)&0xFF;
  mscbuff[4]=(unsigned char)(remtick>>8 )&0xFF;
  mscbuff[5]=(unsigned char)remtick & 0xFF;
}

void SetMsgProcessExit()
{
  mscbuff[0]=D_MSC_EXITPROCESS;
  mscbuff[1]=8;
  mscbuff[2]=(unsigned char) (curpid >> 24) & 0xFF; /*the current process id*/
  mscbuff[3]=(unsigned char) (curpid >> 16) & 0xFF;
  mscbuff[4]=(unsigned char) (curpid >> 8) & 0xFF;
  mscbuff[5]=(unsigned char) curpid & 0xFF;
  mscbuff[6]=(unsigned char)(traceTick>>24)&0xFF;
  mscbuff[7]=(unsigned char)(traceTick>>16)&0xFF;
  mscbuff[8]=(unsigned char)(traceTick>>8 )&0xFF;
  mscbuff[9]=(unsigned char)traceTick & 0xFF;
}

#endif /*_SDLDEBUG*/
